// Smart home energy monitoring and management system(SHEMMS)

#include<math.h>//Link with -lm
#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<unistd.h>//UNIX standard function definitions 
#include<pthread.h>//Link with -lpthread
#include<fcntl.h>//File control definitions
#include<errno.h>//Error number definitions
#include<error.h>
#include<termios.h>//POSIX terminal control definitions
#include<sys/ioctl.h>
#include<stdint.h>
#include<sys/socket.h>
#include<sys/un.h>
#include<sys/types.h>
#include<netinet/in.h>
#include<netinet/ip.h>
#include<sys/stat.h>//to get the status of the a file
#include<sqlite3.h>//Link with -l sqlite3
#include<sys/time.h>
#include<time.h>
#include<signal.h>

#include"sqlite_func.h"
#include"uart_cs5490.h"
#include"gpio_func.h"

#define SERIAL_DEVICE_M1 "/dev/ttyO4" //Serial port for METER1
#define SERIAL_DEVICE_M2 "/dev/ttyO2" //Serial port for METER2

//########################################################################################## [User defined variable structures]
struct server_param{//structure to load data into server thread
	uint8_t ip_address[16];//IP addres to bind
	uint16_t port;//Port number to bind
	uint8_t max_clients;//MAX clients supported in server queue
};

sqlite3 *db1,*db2;
int serial_m1_fd,serial_m2_fd;//maintaining global file desciptor for two serial ports access by all functions
int sfd,cfd;//server,client file descriptor
double energy1=0,energy2=0;//for meter energy readings

/*
struct meter_param{//structure to hold energy meter parameters
	unsigned int id;
	unsigned char* date_time;
	float voltage;
	float current;
	float act_power;
	float rea_power;
	float app_power;
	float pf;
	unsigned int freq;
	float temp;	
};
*/

struct meter_param m1,m2;

//########################################################################################################## [GLOBAL VARIABLES]
volatile uint8_t timer_enabled = 1; //MAIN TIMER CONTROL : 1-Enabled,0-Disabled
volatile uint8_t delay_sec = 1;	//MAIN TIMER CONTROL : Time in seconds (0-255)[1sec reference : should not be touched]	
volatile uint8_t db_update_delay = 5;//MAIN TIMER CONTROL : Database update delay,Time in seconds (0-255)
volatile uint8_t en_update_delay = 2;//MAIN TIMER CONTROL : Energy variable update delay,Time in seconds (0-255)

volatile uint16_t MAX_VOLT = 265;//HIGH VOLTAGE CUTOFF 
volatile uint16_t MIN_VOLT = 170;//LOW VOLTAGE CUTOFF 
volatile float MAX_CURR = 10.0;//HIGH CURRENT CUTOFF 

volatile uint8_t swpolling_enabled = 1;//SWITCH POLLING CONTROL : 1-Enabled,0-Disabled 
volatile uint32_t swpolling_delay_usec = 20000;//SWITCH POLLING DELAY CONTROL : Time in microseconds (0-(2^32-1)) 

volatile uint8_t itervative_server = 1;//MAIN SERVER CONTROL : 0-NON_ITERATIVE(Single execution),1-ITERATIVE[For debugging purpose]
volatile uint8_t client_update_delay = 2;//MAIN TIMER CONTROL : Energy variable update delay,Time in seconds (0-255)
struct server_param myserver = {"127.0.0.1",5211,5};//Server bind IP address,and port number

//########################################################################################################## HELPER FUNCTIONS
uint8_t *get_month_year()//get table name in "month_year" format according to present monnth-year
{
		//Make "month_year" string for table name
		uint8_t *table_name,*dy = get_date_time();//FORMAT = Tue Jul 29 14:39:02 2014
		uint8_t year[5]={0},month[4]={0};

		table_name = (uint8_t*)calloc(10,sizeof(uint8_t));
		strncpy(year,(dy+(strlen(dy)-4)),4);//extract year	
		strncpy(month,(dy+4),3);//extract month
		//printf("Year = %s,Month = %s\n",year,month);
		
		//now create table name as "month_year"
		strcat(table_name,month);
		strcat(table_name,"_");
		strcat(table_name,year);
	
		return table_name;
}

//########################################################################################################### THREAD DEFINATIONS
//-------------------------------------------------------------TIMER THREAD
void *timer_func(void *delay)
{	
	int8_t t_en = en_update_delay, t_db = db_update_delay ;

	printf("TIMER THREAD : Delay programmed = %u\n",*(uint8_t*)delay);
	while(1){	
		while(1==timer_enabled)//loop,only if timer is enabled
		{
			if(0 == t_en)//time for energy update
			{
				t_en =  en_update_delay;//reset timer
				
				if(1==gpio_read(3,21))//if LOAD1 is ON
				{
					if( 0==calc_meter(serial_m1_fd,&m1) );//read meter parameters in m1 global structure,if successful
					{
						if(m1.voltage>MAX_VOLT || m1.voltage<MIN_VOLT)
						{
							printf("**** IRREGULAR VOLTAGE DETECTED ****!!\n");
							gpio_write(3,21,0);//switch off relay1
							printf("Load1 switched OFF!!\n");
							energy1 = 0.0;
						}
						
						if(m1.current>MAX_CURR)
						{
							printf("**** OVERCURRENT DETECTED ****!!\n");
							gpio_write(3,21,0);//switch off relay1
							printf("Load1 switched OFF!!\n");
							energy1 = 0.0;	
						}
						
						energy1 += (m1.act_power/(3600/en_update_delay)); //update energy

						printf("**********************************METER 1\n");
						printf("Voltage(Volt) = %3.2f\n",m1.voltage);
						printf("Current(Ampere) = %3.2f\n",m1.current);
						printf("Active power(Watts)= %3.2f\n",m1.act_power);		
						printf("Reactive power(Watts)= %3.2f\n",m1.rea_power);		
						printf("Apparent power(Watts)= %3.2f\n",m1.app_power);		
						printf("PF = %3.2f\n",m1.pf);		
						printf("Frequency(Hz) = %u\n",m1.freq);
						printf("Energy consumed(Watts-Hour) = %.3f \n",energy1);
					}	
				}
				
				if(1==gpio_read(1,28))//if LOAD2 is ON
				{
					if( 0==calc_meter(serial_m2_fd,&m2) );//read meter parameters in m2 global structure,if successful
					{
						if(m2.voltage>MAX_VOLT || m2.voltage<MIN_VOLT)
						{
							printf("**** IRREGULAR VOLTAGE DETECTED ****!!\n");
							gpio_write(1,28,0);//switch off relay2
							printf("Load2 switched OFF!!\n");
							energy2 = 0.0;
						}
						
						if(m2.current>MAX_CURR)
						{
							printf("**** OVERCURRENT DETECTED ****!!\n");
							gpio_write(1,28,0);//switch off relay2
							printf("Load2 switched OFF!!\n");		
							energy2 = 0.0;
						}

						energy2 += (m2.act_power/(3600/en_update_delay)); //update energy

						printf("*******************************************************METER 2\n");
						printf("Voltage(Volt) = %3.2f\n",m2.voltage);
						printf("Current(Ampere) = %3.2f\n",m2.current);
						printf("Active power(Watts) = %3.2f\n",m2.act_power);		
						printf("Reactive power(Watts) = %3.2f\n",m2.rea_power);		
						printf("Apparent power(Watts) = %3.2f\n",m2.app_power);		
						printf("PF = %3.2f\n",m2.pf);		
						printf("Frequency(Hz) = %u\n",m2.freq);
						printf("Energy consumed(Watts-Hour) = %.3f\n",energy2);
					}
				}
		
				
			}

			if(0 == t_db)//time for database update
			{
				t_db = db_update_delay;//reset db timer

				if(1==gpio_read(3,21))//if LOAD1 is ON
				{
					if( 0==calc_meter(serial_m1_fd,&m1) );//read meter parameters in m1 global structure,if successful
					{
						uint8_t *table_name = get_month_year();//get table name
						m1.id = 1;
						m1.date_time = get_date_time();
						db_insert(db1,table_name,&m1);
						free(table_name);
					}
				}

				if(1==gpio_read(1,28))//if LOAD2 is ON
				{
					if( 0==calc_meter(serial_m2_fd,&m2) );//read meter parameters in m1 global structure,if successful
					{
						uint8_t *table_name = get_month_year();//get table name
						m2.id = 2;
						m2.date_time = get_date_time();
						db_insert(db2,table_name,&m2);
						free(table_name);
					}
				}
				
			}

			sleep(*(uint8_t*)delay);//delay for programmed value of seconds
			t_en--; 
			t_db--;
		}
	}

	pthread_exit("TIMER THREAD : Thread finished successfully!\n");
}

//-------------------------------------------------------------SWITCH POLLING THREAD
void *swpolling_func(void *delay_usec)
{
	printf("SWITCH POLLING THREAD : Delay programmed = %u\n",*(uint32_t*)delay_usec);
	while(1){
		while(1==swpolling_enabled)//loop,only if timer is enabled
		{
			//printf("SWITCH POLLING THREAD : Delay = %u\n",*(uint32_t*)delay_usec);
			if(0==gpio_read(1,17))//read load switch 1
			{
				printf("Load1 switch pressed!!\n");
				sleep(1);
				if(0==gpio_read(3,21))//if switched off previously
				{
					gpio_write(3,21,1);//switch on relay1
					printf("Load1 switched ON!!\n");
				}
				else//if switched on previously
				{
					gpio_write(3,21,0);//switch off relay1
					printf("Load1 switched OFF!!\n");
					energy1 = 0.0;	
				}
			}

			if(0==gpio_read(1,16))//read load switch 2
			{
				printf("Load2 switch pressed!!\n");
				sleep(1);
				if(0==gpio_read(1,28))//if switched off previously
				{
					gpio_write(1,28,1);//switch on relay2
					printf("Load2 switched ON!!\n");
				}
				else//if switched on previously
				{
					gpio_write(1,28,0);//switch off relay2
					printf("Load2 switched OFF!!\n");
					energy2 = 0.0;
				}

			}
			
			usleep(*(uint32_t*)delay_usec);//delay for programmed value of micro seconds
		}
	}

	pthread_exit("SWITCH POLLING THREAD : Thread finished successfully!\n");
}
//-------------------------------------------------------------SERVER THREAD
void *server_func(void *s_param)
{
	printf("SERVER THREAD : IP programmed = %s\n",((struct server_param*)s_param)->ip_address);
	printf("SERVER THREAD : Port number programmed = %u\n",((struct server_param*)s_param)->port);
	printf("SERVER THREAD : MAX clients supported in queue = %u\n",((struct server_param*)s_param)->max_clients);
	
	//unlink(“server_socket”);
	uint8_t rxbuff[50],txbuff[100];
	int32_t res,client_len,server_len;
	struct sockaddr_in server_adr,client_adr;
	
	server_adr.sin_family=AF_INET;
	server_adr.sin_addr.s_addr=inet_addr(((struct server_param*)s_param)->ip_address);
	server_adr.sin_port=((struct server_param*)s_param)->port;

	server_len = sizeof(server_adr);
	client_len = sizeof(client_adr);
		
	//creating a socket
	sfd = socket(AF_INET,SOCK_STREAM,0);//Connection oriented TCP/IP socket	
	if(sfd<0)
	{
		perror("SERVER THREAD : XXXX Unable to open socket : ");
		goto cl_bas;
	}	
	printf("SERVER THREAD : Socket created successfully\n");
	
	uint8_t on = 1;
	setsockopt(sfd,SOL_SOCKET,SO_REUSEADDR,&on,sizeof(on));
	on = 1;
	setsockopt(sfd,SOL_SOCKET,SO_REUSEADDR,&on,sizeof(on));

	//assigning a name(parameters) to a socket
	res = bind(sfd,(struct sockaddr*)&server_adr,server_len);
	if(res<0)
	{
		perror("SERVER THREAD : XXXX Unable to bind socket : ");
		goto cl_ser;
	}
	printf("SERVER THREAD : Socket binded successfully\n");

	//listen to the clients
	res = listen(sfd,((struct server_param*)s_param)->max_clients);
	if(res<0)
	{
		perror("SERVER THREAD : XXXX Unable to listen clients : \n");
		goto cl_ser;
	}

	while(1){
			
		printf("SERVER THREAD : Socket listening to clients now\n");

		/* The argument addr is a pointer to a sockaddr structure.  This structure is filled in with the address of the peer socket, 			as known to the communications layer.  The exact format of the address returned addr is determined by the socket's address 			family .When addr is NULL, nothing is filled in; in this case, addrlen is not used, and should also be NULL.
	      
		The addrlen argument is a value-result argument: the caller must initialize it to contain  the  size  (in  bytes)  
		of  the  structure pointed to by addr; on return it will contain the actual size of the peer address.
		The  returned  address  is  truncated if the buffer provided is too small; in this case, addrlen will return a value greater 			than was supplied to the call.
	
		On  success, these system calls return a nonnegative integer that is a descriptor for the accepted socket.*/
		cfd=accept(sfd,(struct sockaddr *)&client_adr,&client_len);
		if(cfd<0)
		{
			perror("SERVER THREAD : XXXX Unable to accept the client : \n");
			goto cl_ser;
		}

		printf("******************************* CLIENT CONNECTED\n");
		printf("Client connected successfully!!\n"); 
		printf("CLIENT IP ADDRESS : %s\n",inet_ntoa(client_adr.sin_addr.s_addr)); 
		printf("************************************************\n");
		
		while(res!=-1)//send samples continously 
		{	
			sprintf(txbuff,"A%3.2fB%3.2fC%3.2fD%3.2fE%3.2fX%3.2fY%3.2fZ%3.2fS"
						,m1.voltage,m1.current,m1.act_power,energy1
						,m2.voltage,m2.current,m2.act_power,energy2);
			//write to client
			res = write(cfd,txbuff,strlen(txbuff));
			if(res<0)
			{
				perror("SERVER THREAD : Unable to write to the client : \n");
				return -1;
			}	
			else{
				printf("SERVER THREAD : Socket writing to clients now\n");
				printf("SERVER THREAD : Data sent by server : %s\n",txbuff);
			}
			sleep(en_update_delay);//wait for "en_update_delay" seconds
		}
		
		/*
		//read from client
		res = read(cfd,rxbuff,strlen("CLIENT DATA : ABHISHEK d aweome guy :)"));
		if(res<0)
		{
			perror("SERVER THREAD : Unable to read the client : \n");
			return -1;
		}	
		else{
			printf("SERVER THREAD : Socket reading from clients now\n");
			printf("SERVER THREAD : Data read by server : %s\n",rxbuff);
		}
		*/

		res = close(cfd);
		if(res<0)
		{
			perror("SERVER THREAD : Unable to close the client session socket : \n");
			goto cl_cli;
		}

	}
	res = close(sfd);
	if(res<0)
	{
		perror("SERVER THREAD : Unable to close the server listening socket : \n");
		goto cl_ser;
	}
	
	pthread_exit("SERVER THREAD : Thread finished successfully!\n");//if successful,exit from here

	//#### ON ERROR,follow this procedure to exit from thread
cl_cli:	close(cfd);
cl_ser:	close(sfd);
cl_bas:	pthread_exit("SERVER THREAD : Server processing failed!!\n");

}

//#################################################################################### Emergency resource deallocation func
void * res_dealloc(int sig)
{
	int res;
	printf("Emergency resource deallocation func called!!\n");

	res = close(cfd);//close client fd
	if(res<0)
	{
		perror("SERVER THREAD : Unable to close the server listening socket : \n");
	}

	res = close(sfd);//close sever fd
	if(res<0)
	{
		perror("SERVER THREAD : Unable to close the server listening socket : \n");
	}

	res = close(serial_m1_fd);//close serial file descriptor
	if(res<0)
	{
		perror("XXXX : Unable to close serial device - ");
	}
	
	res = close(serial_m2_fd);//close serial file descriptor
	if(res<0)
	{
		perror("XXXX : Unable to close serial device - ");
	}
	
	exit(0);
}


//########################################################################################################### MAIN PROGRAM START
int main(int argc,uint8_t *argv[])//for taking SERVER binding IP address from command line
{
	struct meter_param *mp;

	int res;//temporary variable to store API results
	pthread_t timer_thread, server_thread, swpolling_thread;//thread ID holding variables
	void *timer_back_result, *server_back_result, *swpolling_back_result;//thread return value pointing variable
	
	if(2==argc && strlen(argv[1])<=15){//if a valid ip address is received through command line
		strcpy(myserver.ip_address,argv[1]);//get server binding ip address,else use default ip address
	}

	//-----------------------------------------------------------------------------------Welcome message
	printf("## Welcome to Smart home energy monitoring and management system(SHEMMS) gateway ##\n");
	printf("@=> Date-Time today : %s\n",get_date_time());
	printf("@=> Server IP address  : %s\n\n",myserver.ip_address);
	

	//-------------------------------Initialize signal handlers for emergency stop resource deallocation
	signal(SIGINT,res_dealloc);
	signal(SIGTSTP,res_dealloc);
	
	//---------------------------------------------------------------------Initialize all gpio pins used
	gpio_init(1,16);//Load 2 switch
	gpio_dir(1,16,1);//set gpio1_16(15 pin) = input
	
	gpio_init(1,28);//Load 2 relay
	gpio_dir(1,28,0);//set gpio1_28(12 pin) = output
	gpio_write(1,28,0);//switch off relay
	
	gpio_init(1,17);//Load 1 switch
	gpio_dir(1,17,1);//set gpio1_17(23 pin) = input
	
	gpio_init(3,21);//Load 1 relay
	gpio_dir(3,21,0);//set gpio3_21(21 pin) = output
	gpio_write(3,21,0);//switch off relay
	
	//---------------------------------------------------------------------SERIAL PORTS INIT & CONFIG
	serial_m1_fd = open_port(SERIAL_DEVICE_M1);//open the port and assign file descriptor
	if(serial_m1_fd<0)
	{
		printf("XXXX : Unable to open serial port \"%s\" !\n",SERIAL_DEVICE_M1);
	}	
	
	res = set_config_uart(serial_m1_fd);//initialize setting for the particular opened port
	if(res<0)
	{
		printf("XXXX : Unable to set configurations for serial port \"%s\" !\n",SERIAL_DEVICE_M1);
	}
	
	serial_m2_fd = open_port(SERIAL_DEVICE_M2);//open the port and assign file descriptor
	if(serial_m2_fd<0)
	{
		printf("XXXX : Unable to open serial port \"%s\" !\n",SERIAL_DEVICE_M2);
	}	

	res = set_config_uart(serial_m2_fd);//initialize setting for the particular opened port
	if(res<0)
	{
		printf("XXXX : Unable to set configurations for serial port \"%s\" !\n",SERIAL_DEVICE_M2);
	}

	//---------------------------------------------------------------------CS5490 IC INIT & CONFIG
	if(0==set_calibration_constants(serial_m1_fd))
		printf("Calibration constants loaded,energy conversion started on Meter1!\n");
	else
		printf("XXXX : Calibration failed,conversion not started on Meter1!\n");
	
	if(0==set_calibration_constants(serial_m2_fd))
		printf("Calibration constants loaded,energy conversion started on Meter2!\n");
	else
		printf("XXXX : Calibration failed,conversion not started on Meter2!\n");
	
	//---------------------------------------------------------------------SQLite database initialization
	db1 = db_open("METER1");//create/open METER1 database
	if(db1!=NULL)//success
	{
		uint8_t *table_name = get_month_year();
		db_create_table(db1,table_name);//now create table if it is not there for current month_year
		free(table_name);
	}
	
	db2 = db_open("METER2");//create/open METER1 database
	if(db2!=NULL)//success
	{
		uint8_t *table_name = get_month_year();
		db_create_table(db2,table_name);//now create table if it is not there for current month_year
		free(table_name);
	}	

	//--------------------------------------------------------------------- MULTIPLE THREAD CREATION
	// Create main timer thread : TIMER THREAD
	res = pthread_create(&timer_thread,NULL,timer_func,&delay_sec);
	if(0!=res)
	{
		perror("XXXX : Error creating thread - ");
		exit(EXIT_FAILURE);
	}
	printf("Thread creation successful : TIMER THREAD [ID = %u]\n",timer_thread);

	// Create server thread : SERVER THREAD
	res = pthread_create(&server_thread,NULL,server_func,&myserver);
	if(0!=res)
	{
		perror("XXXX : Error creating thread - ");
		exit(EXIT_FAILURE);
	}
	printf("Thread creation successful : SERVER THREAD [ID = %u]\n",server_thread);
	
	// Create switches polling thread : SWITCH POLLING THREAD
	res = pthread_create(&swpolling_thread,NULL,swpolling_func,&swpolling_delay_usec);
	if(0!=res)
	{
		perror("XXXX : Error creating thread - ");
		exit(EXIT_FAILURE);
	}
	printf("Thread creation successful : SWITCH POLLING THREAD [ID = %u]\n",swpolling_thread);
	
	//--------------------------------------------------------------------- THREADS JOINING
	// Join thread : SERVER THREAD
	res = pthread_join(server_thread,&server_back_result);
	if(0!=res)
	{
		perror("XXXX : Thread joining failed - ");
		//exit(EXIT_FAILURE);
	}
	printf("Thread joined with msg = %s\n",(char*)server_back_result);

	// Join thread : TIMER THREAD
	res = pthread_join(timer_thread,&timer_back_result);
	if(0!=res)
	{
		perror("XXXX : Thread joining failed - ");
		//exit(EXIT_FAILURE);
	}
	printf("Thread joined with msg = %s\n",(char*)timer_back_result);

	// Join thread : SWITCH POLLING THREAD
	res = pthread_join(swpolling_thread,&swpolling_back_result);
	if(0!=res)
	{
		perror("XXXX : Thread joining failed - ");
		//exit(EXIT_FAILURE);
	}
	printf("Thread joined with msg = %s\n",(char*)swpolling_back_result);
	

	//--------------------------------------------------------------------- Resources deallocation
	//deinitialize all gpio pins used
	gpio_de_init(1,16);
	gpio_de_init(1,28);
	gpio_de_init(1,17);
	gpio_de_init(3,21);
	
	res = close(serial_m1_fd);//close serial file descriptor
	if(res<0)
	{
		perror("XXXX : Unable to close serial device - ");
	}
	
	res = close(serial_m2_fd);//close serial file descriptor
	if(res<0)
	{
		perror("XXXX : Unable to close serial device - ");
	}

	return 0;//depict program execution successful
}	
//########################################################################################################### MAIN PROGRAM EXIT

